/* ------------------------------------------------------------------------------
  File: chr6dm_HMC.c
  Author: CH Robotics
  Version: 1.0
  
  Description: Functions for interacting with HMC5843 magnetic sensor
------------------------------------------------------------------------------ */ 

#include "stm32f10x.h"
#include "CHR_i2c.h"
#include "UM6_usart.h"
#include "UM6_startup.h"
#include "CHR_delays.h"

// Buffers for storing i2c data
uint8_t g_i2cTxBuf[I2C_TX_BUFSIZE];
uint8_t g_i2cRxBuf[I2C_RX_BUFSIZE];
uint8_t g_i2cBytesToTransmit;
uint8_t g_i2cBytesToReceive;
uint8_t g_i2cReadPrelude;
__IO uint8_t g_i2cTxPtr;
__IO uint8_t g_i2cRxPtr;
__IO uint8_t g_i2cBusy;
__IO uint8_t g_i2cBusError;
__IO uint8_t g_i2cDirection;
__IO uint8_t g_i2cCurrentSlaveAddress;

/*******************************************************************************
* Function Name  : i2cBufWrite
* Input          : uint8_t addr, uint8_t* i2cData, uint8_t length
* Output         : None
* Return         : 1 if success, 0 if fail
* Description    : Writes 'length' bytes from i2cData[] to the i2c address 'addr'
						 This is a blocking function.
*******************************************************************************/
int32_t i2cBufWrite( uint8_t addr, uint8_t* i2cData, uint8_t length )
{
	 int32_t index;
	 	 
	 // Send START condition
	 I2C_GenerateSTART(I2C1, ENABLE);

	 // Wait for START condition to transmit
	 if( !i2cWaitForEvent(I2C_EVENT_MASTER_MODE_SELECT) )
	 {
		  i2cReset();
//		  I2C_GenerateSTOP(I2C1, ENABLE);
		  return 0;
	 }
	 
	 // Send slave address
	 I2C_Send7bitAddress(I2C1, addr, I2C_Direction_Transmitter);

	 // Wait for ACK
	 if( !i2cWaitForEvent(I2C_EVENT_MASTER_TRANSMITTER_MODE_SELECTED) )
	 {
		  i2cReset();
//		  I2C_GenerateSTOP(I2C1, ENABLE);
		  return 0;
	 }

	 // Start transmitting data
	 for( index = 0; index < length; index++ )
	 {
		  // Send byte
		  I2C_SendData(I2C1, i2cData[index]);

		  // Wait for ACK
		  if( !i2cWaitForEvent(I2C_EVENT_MASTER_BYTE_TRANSMITTED) )
		  {
				i2cReset();
//				I2C_GenerateSTOP(I2C1, ENABLE);
				return 0;
		  }
	 }
	 
	 /* Send STOP condition */
	 I2C_GenerateSTOP(I2C1, ENABLE);
	 
	 return 1;
}


/*******************************************************************************
* Function Name  : i2cRead
* Input          : uint8_t addr, uint8_t* i2cData, uint8_t length
* Output         : None
* Return         : 1 if success, 0 if fail
* Description    : Writes 'length' bytes from i2cData[] to the i2c address 'addr'
						 This is a blocking function.
*******************************************************************************/
int32_t i2cRead( uint8_t addr, uint8_t* i2cData, uint8_t bytesToRead )
{
	 uint8_t* pBuffer = i2cData;
	 int32_t retries;
	 
	 // Send START condition
	 I2C_GenerateSTART(I2C1, ENABLE);

	 // Wait for START condition to transmit
	 if( !i2cWaitForEvent(I2C_EVENT_MASTER_MODE_SELECT) )
	 {
		  i2cReset();
//		  I2C_GenerateSTOP(I2C1, ENABLE);
		  return 0;
	 }
	 
	 // Send slave address
	 I2C_Send7bitAddress(I2C1, addr, I2C_Direction_Receiver);

	 // Wait for ACK
	 if( !i2cWaitForEvent(I2C_EVENT_MASTER_RECEIVER_MODE_SELECTED) )
	 {
		  i2cReset();
//		  I2C_GenerateSTOP(I2C1, ENABLE);
		  return 0;
	 }
	 
	 // Start receiving data
	 retries = 0;
	 while(bytesToRead  && (retries < MAX_RETRIES))  
	 {
		  if(bytesToRead == 1)
		  {
				// Disable Acknowledgement
				I2C_AcknowledgeConfig(I2C1, DISABLE);

				// Send STOP Condition 
//				I2C_GenerateSTOP(I2C1, ENABLE);
				i2cReset();
		  }

		  /* Test on EV7 and clear it */
		  if(I2C_CheckEvent(I2C1, I2C_EVENT_MASTER_BYTE_RECEIVED))  
		  {      
				// Read a byte from the EEPROM
				*pBuffer = I2C_ReceiveData(I2C1);

				/* Point to the next location where the byte read will be saved */
				pBuffer++; 

				/* Decrement the read bytes counter */
				bytesToRead--;
				
				retries = 0;
		  }
		  
		  retries++;
	 }
	 
	 if( retries >= MAX_RETRIES )
	 {
//		  I2C_GenerateSTOP(I2C1, ENABLE);
		  i2cReset();
		  return 0;
	 }
	 
	 
	 /* Enable Acknowledgement to be ready for another reception */
	 I2C_AcknowledgeConfig(I2C1, ENABLE);
	 
	 return 1;
}


/*******************************************************************************
* Function Name  : i2cWaitForEvent
* Input          : uint32_t event
* Output         : None
* Return         : 1 if success, 0 if fail
* Description    : Wrapper function for i2c functionality.  Checks for the specified
						 event I2C_ACK_ATTEMPTS times, and then errors out if the event
						 hasn't occured.
*******************************************************************************/
int i2cWaitForEvent( uint32_t event )
{
	 int32_t attempts;
	 
	 attempts = 0;
	 while(!I2C_CheckEvent(I2C1, event))
	 {
		  attempts++;
		  if( attempts > I2C_ACK_ATTEMPTS )
		  {
				return 0;
		  }		  
	 }
	 
	 return 1;
}

/*******************************************************************************
* Function Name  : i2cStart
* Input          : None
* Output         : None
* Return         : 1 if success, 0 if fail
* Description    : 

Starts the i2c transmission by setting the start sequence on the bus.  Before
this function is called, the data structures specifying what the i2c
transfer is should be set.

- If an i2c write, this buffer should be filled with the data to write
   uint8_t g_i2cTxBuf[I2C_TX_BUFSIZE];
- These should be set
   uint8_t g_i2cBytesToTransmit;
   uint8_t g_i2cBytesToReceive;
- These needn't be set explicitly - they will be set to zero by this function
   __IO uint8_t g_i2cTxPtr;
   __IO uint8_t g_i2cRxPtr;
- This needn't be set explicitly - it will be handles by this function
   __IO uint8_t g_i2cBusy;
- This should be set to either I2C_TRANSMITTER or I2C_RECEIVER
   __IO uint8_t g_i2cDirection;
- This should be set to the slave address being accessed on the i2c bus
   __IO uint8_t g_i2cCurrentSlaveAddress;

*******************************************************************************/
int32_t i2cStart()
{
	 DMA_InitTypeDef DMA_InitStructure;
	 
	 if( g_i2cBusy )
	 {
		  return 0;
	 }
	 
	 g_i2cBusy = 1;
	 g_i2cBusError = 0;
	 g_i2cTxPtr = 0;
	 g_i2cRxPtr = 0;
	 
/*		OLD CODE - THIS WAS USED WITH AN INTERRUPT-ONLY I2C CONFIGURATION
	 // Start i2c timeout timer - if this timer fires, then there was an i2c bus error.
	 // The bus will be restarted by the event handler.
	 TIM_SetCounter(TIM4,0);
	 TIM_Cmd( TIM4, ENABLE );
	 
	 // Make sure acknowledgements are enabled for read operations
	 // (these are disabled on the last byte read from a device, so they need to be re-enabled
	 // before another read can be performed)
	 I2C_AcknowledgeConfig(I2C1, ENABLE);
	 
	 // Generate i2c start condition
	 I2C_GenerateSTART(I2C1, ENABLE);
*/
	 
	 // Configure i2c for DMA transfer
	 if( g_i2cDirection == I2C_TRANSMITTER )
	 {
		  // i2c TX DMA initialization
		  DMA_DeInit(DMA1_Channel6);
		  DMA_InitStructure.DMA_PeripheralBaseAddr = (uint32_t)I2C1_DR_Address;
		  DMA_InitStructure.DMA_MemoryBaseAddr = (uint32_t)g_i2cTxBuf;
		  DMA_InitStructure.DMA_DIR = DMA_DIR_PeripheralDST;
		  DMA_InitStructure.DMA_BufferSize = g_i2cBytesToTransmit;
		  DMA_InitStructure.DMA_PeripheralInc = DMA_PeripheralInc_Disable;
		  DMA_InitStructure.DMA_MemoryInc = DMA_MemoryInc_Enable;
		  DMA_InitStructure.DMA_PeripheralDataSize = DMA_MemoryDataSize_Byte;
		  DMA_InitStructure.DMA_MemoryDataSize = DMA_MemoryDataSize_Byte;
		  DMA_InitStructure.DMA_Mode = DMA_Mode_Normal;
		  DMA_InitStructure.DMA_Priority = DMA_Priority_VeryHigh;
		  DMA_InitStructure.DMA_M2M = DMA_M2M_Disable;
		  
		  DMA_Init(DMA1_Channel6, &DMA_InitStructure);
		  
		  I2C_AcknowledgeConfig(I2C1, ENABLE);
		  
//		  DMA_ITConfig( DMA1_Channel6, DMA_IT_TC, ENABLE );
	 }
	 else
	 {
		  // i2c RX DMA initialization
		  DMA_DeInit(DMA1_Channel7);
		  DMA_InitStructure.DMA_PeripheralBaseAddr = (uint32_t)I2C1_DR_Address;
		  DMA_InitStructure.DMA_MemoryBaseAddr = (uint32_t)g_i2cRxBuf;
		  DMA_InitStructure.DMA_DIR = DMA_DIR_PeripheralSRC;
		  DMA_InitStructure.DMA_BufferSize = g_i2cBytesToReceive;
		  DMA_InitStructure.DMA_PeripheralInc = DMA_PeripheralInc_Disable;
		  DMA_InitStructure.DMA_MemoryInc = DMA_MemoryInc_Enable;
		  DMA_InitStructure.DMA_PeripheralDataSize = DMA_MemoryDataSize_Byte;
		  DMA_InitStructure.DMA_MemoryDataSize = DMA_MemoryDataSize_Byte;
		  DMA_InitStructure.DMA_Mode = DMA_Mode_Normal;
		  DMA_InitStructure.DMA_Priority = DMA_Priority_VeryHigh;
		  DMA_InitStructure.DMA_M2M = DMA_M2M_Disable;
		  
		  DMA_Init(DMA1_Channel7, &DMA_InitStructure);
		  
		  if( g_i2cBytesToReceive == 1 )
		  {
				I2C_AcknowledgeConfig(I2C1, DISABLE);
		  }
		  else
		  {
				I2C_AcknowledgeConfig(I2C1, ENABLE);
		  }
		  
		  // Enable LAST bit in I2C_CR2 register so that a NACK is generated on the last data read
		  I2C_DMALastTransferCmd( I2C1, ENABLE );
				
//		  DMA_ITConfig( DMA1_Channel7, DMA_IT_TC, ENABLE );
	 }
	 
	 // Generate start condition and wait for response
	 I2C_GenerateSTART(I2C1, ENABLE);
	
	 if( !i2cWaitForEvent( I2C_EVENT_MASTER_MODE_SELECT ) )
	 {
		  i2cStop();
		  i2cReset();
		  return 0;
	 }
	 
	 // Send I2C2 slave Address for write
	 I2C_Send7bitAddress(I2C1, g_i2cCurrentSlaveAddress, g_i2cDirection);
	 	 
	 if( g_i2cDirection == I2C_TRANSMITTER )
	 {
		  if( !i2cWaitForEvent( I2C_EVENT_MASTER_TRANSMITTER_MODE_SELECTED ) )
		  {
				i2cStop();
				i2cReset();
				return 0;
		  }
		  
		  // Enable the DMA controller to transfer the data
		  I2C_DMACmd(I2C1, ENABLE);
		  DMA_Cmd(DMA1_Channel6, ENABLE);
		  
		  // Wait for DMA transfer to complete
		  while(!DMA_GetFlagStatus(DMA1_FLAG_TC6));
				
		  // Wait for last byte to transfer
		  while( !I2C_GetFlagStatus( I2C1, I2C_FLAG_BTF ) );		  
	 }
	 else
	 {
		  if( !i2cWaitForEvent( I2C_EVENT_MASTER_RECEIVER_MODE_SELECTED ) )
		  {
				i2cStop();
				i2cReset();
				return 0;
		  }
		  
		  // Enable the DMA controller to transfer the data
		  I2C_DMACmd(I2C1, ENABLE);
		  DMA_Cmd(DMA1_Channel7, ENABLE);
		  
		  // Wait for DMA transfer to complete
		  while(!DMA_GetFlagStatus(DMA1_FLAG_TC7));
	 }
	 
	 // Send I2C1 STOP Condition
	 if( !g_i2cReadPrelude )
	 {
		  I2C_GenerateSTOP(I2C1, ENABLE);
	 }
	 
	 i2cStop();
	 
	 return 1;
}

/*******************************************************************************
* Function Name  : i2cStop
* Input          : None
* Output         : None
* Return         : 1 if success, 0 if fail
* Description    : 

This function is called by the i2c event handler when an i2c transfer is
complete.

*******************************************************************************/
void i2cStop()
{
	 g_i2cBusy = 0;
	 
	 // Stop the i2c timeout counter
	 TIM_Cmd( TIM4, DISABLE );
	 
	 return;
}

/*******************************************************************************
* Function Name  : i2cWait
* Input          : None
* Output         : None
* Return         : 1 if success, 0 if fail
* Description    : 

Waits for the i2c bus to become available.  If a bus error occurs, then
return 0;

(the bus error flag is set in an interrupt routine when the i2c timeout timer fires)

*******************************************************************************/
int32_t i2cWait()
{
	 if( g_i2cBusy == 0 )
	 {
		  return 1;
	 }
	 
	 while( g_i2cBusy )
	 {
		  if( g_i2cBusError )
		  {
				g_i2cBusError = 0;
				g_i2cBusy = 0;
				return 0;
		  }
	 }
	 
	 return 1;
}

/*******************************************************************************
* Function Name  : i2cReset
* Input          : None
* Output         : None
* Return         : 1 if success, 0 if fail
* Description    : Resets i2c bus (used on bus errors)
*******************************************************************************/
void i2cReset( )
{
	 I2C_InitTypeDef   I2C_InitStructure;
	 
	 // Stop the i2c timeout counter
	 TIM_Cmd( TIM4, DISABLE );
	 
	 // De-initialize i2c bus
	 I2C_DeInit(I2C1);
	 
	 // Enable I2C
	 I2C_Cmd(I2C1, ENABLE);
	 
	 // I2C1 configuration
	 I2C_InitStructure.I2C_Mode = I2C_Mode_I2C;
	 I2C_InitStructure.I2C_DutyCycle = I2C_DutyCycle_2;
	 I2C_InitStructure.I2C_OwnAddress1 = THIS_SLAVE_ADDRESS7;
	 I2C_InitStructure.I2C_Ack = I2C_Ack_Enable;
	 I2C_InitStructure.I2C_AcknowledgedAddress = I2C_AcknowledgedAddress_7bit;
	 I2C_InitStructure.I2C_ClockSpeed = I2C_CLOCK_SPEED;
	 I2C_Init(I2C1, &I2C_InitStructure);
	 
	 // Enable I2C1 event and buffer interrupts
	 // DISABLED FOR DMA USAGE
//	 I2C_ITConfig(I2C1, I2C_IT_EVT | I2C_IT_BUF, ENABLE);

	 // Reset buffer pointers
	 g_i2cTxPtr = 0;
	 g_i2cRxPtr = 0;
}

void DMA1_Channel6_IRQHandler( void )
{
	 
}

/**
  * @brief  This function handles I2C1 Event interrupt request.
  * @param  None
  * @retval None
  */
void I2C1_EV_IRQHandler(void)
{
  switch (I2C_GetLastEvent(I2C1))
  {
    case I2C_EVENT_MASTER_MODE_SELECT:                 /* EV5 */
      if(g_i2cDirection == I2C_TRANSMITTER)
      {
        /* Master Transmitter ----------------------------------------------*/
        /* Send slave Address for write */
        I2C_Send7bitAddress(I2C1, g_i2cCurrentSlaveAddress, I2C_Direction_Transmitter);
      }
      else
      {
        /* Master Receiver -------------------------------------------------*/
        /* Send slave Address for read */
        I2C_Send7bitAddress(I2C1, g_i2cCurrentSlaveAddress, I2C_Direction_Receiver);
      }
		
		g_i2cBusy = 1;
		
      break;
        
    /* Master Transmitter --------------------------------------------------*/
    /* Test on I2C1 EV6 and first EV8 and clear them */
    case I2C_EVENT_MASTER_TRANSMITTER_MODE_SELECTED: 
      
      /* Send the first data */
      I2C_SendData(I2C1, g_i2cTxBuf[g_i2cTxPtr]);
		g_i2cTxPtr++;
	 
      break;
	 
	 // Test on I2C1 EV8 and clear it
	 // This event occurs when there is data in the i2c shift register, but no data waiting in the buffer
	 // (the shift register contains what is being transmitted, while the data buffer is waiting to be transmitted)
	 // As long as bytes remain to be transmitted, copy the next byte into the data buffer.
	 // If the correct number of bytes have been transmitted (ie. the current byte in the shift register is the last)
	 // then disable the ITC_IT_BUF interrupt.  This will prevent the I2C_EVENT_MASTER_BYTE_TRANSMITTING from being
	 // called.  Then, the I2C_EVENT_MASTER_BYTE_TRANSMITTED event will be called when the current byte is finished transmitting.
	 case I2C_EVENT_MASTER_BYTE_TRANSMITTING:  /* Without BTF, EV8 */     
		  if(g_i2cTxPtr < g_i2cBytesToTransmit)
		  {
				// Add next byte to TX buffer
				I2C_SendData(I2C1, g_i2cTxBuf[g_i2cTxPtr]);
				g_i2cTxPtr++;
		  }
		  else
		  {
				I2C_GenerateSTOP(I2C1, ENABLE);
				i2cStop();
		  }
		  break;

		  // This interrupt handler is called when the i2c shift register has just emptied (ie. a byte just finished transmitting)
		  // and there is no new byte in the data buffer to copy to the shift register.  This should ideally only be called after everything is
		  // finished transmitting.  However, in rare cases where the I2C_EVENT_MASTER_BYTE_TRANSMITTING event was not responded to quickly enough,
		  // the i2c transfer may not be finished.  In that case, copy the next byte to the data buffer.
		  case I2C_EVENT_MASTER_BYTE_TRANSMITTED: /* With BTF EV8-2 */
				if( g_i2cTxPtr < g_i2cBytesToTransmit )
				{
					 /* Transmit I2C1 data */
					 I2C_SendData(I2C1, g_i2cTxBuf[g_i2cTxPtr]);
					 g_i2cTxPtr++;
					 
					 if( g_i2cTxPtr == g_i2cBytesToTransmit )
					 {
						  I2C_GenerateSTOP(I2C1, ENABLE);
						  i2cStop();
					 }					 
				}
				else
				{
					 // If i2c busy flag is still set, then the stop condition has not yet been enabled.  Enable it.
					 if( g_i2cBusy )
					 {
						  I2C_GenerateSTOP(I2C1, ENABLE);
						  i2cStop();
					 }
				}
		  break;

    /* Master Receiver -------------------------------------------------------*/
    case I2C_EVENT_MASTER_RECEIVER_MODE_SELECTED:
      if(g_i2cBytesToReceive == 1)
      {
        /* Disable I2C1 acknowledgement */
        I2C_AcknowledgeConfig(I2C1, DISABLE);
        /* Send I2C1 STOP Condition */
        I2C_GenerateSTOP(I2C1, ENABLE);
		  
		  // Reset the i2c bus to prep for the next transfer
		  i2cStop();
      }
      break;

   /* Test on I2C1 EV7 and clear it */
   case I2C_EVENT_MASTER_BYTE_RECEIVED:
     /* Store I2C1 received data */
     g_i2cRxBuf[g_i2cRxPtr++] = I2C_ReceiveData(I2C1);
     /* Disable ACK and send I2C1 STOP condition before receiving the last data */
     if(g_i2cRxPtr == (g_i2cBytesToReceive - 1))
     {
       /* Disable I2C1 acknowledgement */
       I2C_AcknowledgeConfig(I2C1, DISABLE);
       /* Send I2C1 STOP Condition */
       I2C_GenerateSTOP(I2C1, ENABLE);
			
		 // Reset the i2c bus to prep for the next transfer
		 i2cStop();
     }
     break;

    default:
      break;
  }
}


// -----------------------------------------------------------------------------------------------
// CODE FOR BIT-BANGING THE I2C INTERFACE.
// -----------------------------------------------------------------------------------------------

// Flag to indicate that an i2c transfer has already started.
uint8_t g_i2cStarted = 0;


/*******************************************************************************
* Function Name  : i2cReset
* Input          : None
* Output         : None
* Return         : 1 if success, 0 if fail
* Description    : Resets i2c bus (used on bus errors)
*******************************************************************************/
uint8_t i2cReadSDA( void )
{
	 // Set SDA pin to high-Z state
	 GPIO_WriteBit( I2C_PORT, I2C_SDA_PIN, 1 );
	 
	 // Return the value of the bit
	 return GPIO_ReadInputDataBit( I2C_PORT, I2C_SDA_PIN );
}

void i2cClearSDA( void )
{
	 // Pull SDA pin low
	 GPIO_WriteBit( I2C_PORT, I2C_SDA_PIN, 0 );
}

void i2cClearSCL( void )
{
	 // Pull SCL pin low
	 GPIO_WriteBit( I2C_PORT, I2C_SCL_PIN, 0 );
}

uint8_t i2cReadSCL( void )
{
	 // Set SCL pin to high-Z state
	 GPIO_WriteBit( I2C_PORT, I2C_SCL_PIN, 1 );
	 
	 // Return the value of the bit
	 return GPIO_ReadInputDataBit( I2C_PORT, I2C_SCL_PIN );	 
}

uint8_t i2cGenerateStart( void )
{
	 int32_t attempts;
	 I2C_DELAY_INIT();
	 
	 if( g_i2cStarted )
	 {
		  // Set SDA to 1
		  i2cReadSDA();
		  
		  I2C_DELAY();
		  
		  // Handle clock stretching
		  attempts = 0;
		  while( (i2cReadSCL() == 0) )
		  {
				attempts++;
				
				if( attempts > I2C_MAX_ATTEMPTS )
				{
					 return I2C_TIMEOUT;
				}
		  }
	 }
	 if( i2cReadSDA() == 0 )
	 {
		  // Something is pulling the SDA pin low.  This shouldn't be happening.
		  // Return arbitration lost.
		  return I2C_ARBITRATION_LOST;
	 }
	 
	 // Set SDA from 1 to 0
	 i2cClearSDA();
	 I2C_DELAY();
	 I2C_DELAY();
	 I2C_DELAY();
	 I2C_DELAY();
	 I2C_DELAY();
	 I2C_DELAY();	 
	 i2cClearSCL();
	 
	 // Set flag indicating that an i2c was started
	 g_i2cStarted = 1;
	 
	 return I2C_SUCCESS;
}

uint8_t i2cWriteBit( uint8_t value )
{
	 uint8_t index;
	 int32_t attempts;
	 I2C_DELAY_INIT();
	 
	 if( value == I2C_BIT_SET )
	 {
		  i2cReadSDA();
	 }
	 else
	 {
		  i2cClearSDA();
	 }
	 	 
	 I2C_DELAY();
	 
	 // Set SCL high and handle clock stretching
	 attempts = 0;
	 while( (i2cReadSCL() == 0) )
	 {
		  attempts++;
		  
		  if( attempts > I2C_MAX_ATTEMPTS )
		  {
				return I2C_TIMEOUT;
		  }
	 }
	 
	 // SCL is high - data should be valid
	 // If SDA is supposed to be high, make sure that nothing else on the bus is screwing with it
	 /*
	 if( value == I2C_BIT_SET )
	 {
		  if( i2cReadSDA() == 0 )
		  {
				return I2C_ARBITRATION_LOST;
		  }
	 }
	 */
	 
	 // Delay for half the clock period after read.
	 I2C_DELAY();

	 // Now set SCL low
	 i2cClearSCL();
	 
	 I2C_DELAY();
	 
	 return I2C_SUCCESS;
}

uint8_t i2cReadBit( void )
{
	 int32_t attempts;
	 uint8_t bus_data;
	 I2C_DELAY_INIT();
	 
	 // Let the slave drive the data
	 i2cReadSDA();
	 
	 I2C_DELAY_SHORT();
	 
	 // Handle clock stretching
	 attempts = 0;
	 while( (i2cReadSCL() == 0) )
	 {
		  attempts++;
		  
		  if( attempts > I2C_MAX_ATTEMPTS )
		  {
				return I2C_TIMEOUT;
		  }
	 }
	 
	 // Data should be valid now
//	 i2cReadSDA();
//	 I2C_DELAY_SHORT();
	 bus_data = i2cReadSDA();

	 I2C_DELAY();
	 
	 // Set clock low
	 i2cClearSCL();
	 
	 I2C_DELAY();
	 
	 return bus_data;
}

uint8_t i2cFailed( uint8_t status )
{
	 i2cGenerateStop();
	 return status;
}

uint8_t i2cGenerateStop( void )
{
	 int32_t attempts;
	 I2C_DELAY_INIT();
	 
	 // Set SDA to 0
	 i2cClearSDA();
	 I2C_DELAY();
	 
	 // Handle clock stretching
	 attempts = 0;
	 while( (i2cReadSCL() == 0) )
	 {
		  attempts++;
		  
		  if( attempts > I2C_MAX_ATTEMPTS )
		  {
				return I2C_TIMEOUT;
		  }
	 }
	 
	 // Set SDA from 0 to 1
	 attempts = 0;
	 while( (i2cReadSDA() == 0) )
	 {
		  attempts++;
		  
		  if( attempts > I2C_MAX_ATTEMPTS )
		  {
				return I2C_ARBITRATION_LOST;
		  }
	 }
	 
	 I2C_DELAY();
	 
	 g_i2cStarted = 0;
	 
	 return I2C_SUCCESS;
	 
}

uint8_t i2cTransmitByte( uint8_t byte, uint8_t send_start, uint8_t send_stop )
{
	 uint8_t bit_index;
	 uint8_t nack;
	 uint8_t returnval;
	 
	 if( send_start == I2C_SEND_START )
	 {
		  returnval = i2cGenerateStart();
		  if( returnval != I2C_SUCCESS )
		  {
				return returnval;
		  }
	 }
	 
	 for( bit_index = 0; bit_index < 8; bit_index++ )
	 {
		  returnval = i2cWriteBit((byte >> 7) & 0x01);
		  if( returnval != I2C_SUCCESS )
		  {
				return returnval;
		  }
		  
		  byte <<= 1;
	 }
	 
	 nack = i2cReadBit();
	 
	 if( send_stop == I2C_SEND_STOP )
	 {
		  returnval = i2cGenerateStop();
		  
		  if( returnval != I2C_SUCCESS )
		  {
				return returnval;
		  }
	 }
	 
	 return nack;
}

uint8_t i2cReadByte( uint8_t nack, uint8_t send_stop, uint8_t* status )
{
	 uint8_t byte = 0;
	 uint8_t bit_index;
	 uint8_t returnval;
	 
	 for( bit_index = 0; bit_index < 8; bit_index++ )
	 {
		  byte <<= 1;
		  returnval = i2cReadBit();
		  if( returnval > 1 )
		  {
				*status = returnval;
				return 0;
		  }
		  byte |= returnval;
	 }
	 returnval = i2cWriteBit( nack );
	 if( returnval != I2C_SUCCESS )
	 {
		  return returnval;
	 }
	 
	 if( send_stop == I2C_SEND_STOP )
	 {
		  returnval = i2cGenerateStop();
		  if( returnval != I2C_SUCCESS )
		  {
				*status = returnval;
				return 0;
		  }
	 }
	 
	 *status = I2C_SUCCESS;
	 
	 return byte;	 
}

uint8_t i2cBatchWrite( uint8_t address7, uint8_t* txBuffer, uint8_t bytesToWrite )
{
	 int i;
	 uint8_t returnval;
	 
	 // Send start condition and address byte
	 returnval = i2cTransmitByte( (address7 | 0x00), I2C_SEND_START, I2C_NO_STOP );
	 
	 if( returnval != I2C_SUCCESS )
	 {
		  return i2cFailed(returnval);
	 }
	 
	 // Send data bytes
	 for( i = 0; i < (bytesToWrite-1); i++ )
	 {
		  i2cTransmitByte( txBuffer[i], I2C_NO_START, I2C_NO_STOP );
		  
		  if( returnval != I2C_SUCCESS )
		  {
				return i2cFailed(returnval);
		  }
	 }
	 
	 // Send the final byte along with stop condition
	 returnval = i2cTransmitByte( txBuffer[bytesToWrite-1], I2C_NO_START, I2C_SEND_STOP );
	 
	 return returnval;
}

uint8_t i2cBatchRead( uint8_t address7, uint8_t device_start_address, uint8_t* rxBuffer, uint8_t bytesToRead )
{
	 int i;
	 uint8_t returnval;
	 
	 // Send start condition and address byte
	 returnval = i2cTransmitByte( (address7 | 0x00), I2C_SEND_START, I2C_NO_STOP );
	 
	 if( returnval != I2C_SUCCESS )
	 {
		  return i2cFailed(returnval);
	 }
	 
	 // Send start address for read
	 returnval = i2cTransmitByte( device_start_address, I2C_NO_START, I2C_NO_STOP );
	 
	 if( returnval != I2C_SUCCESS )
	 {
		  return i2cFailed(returnval);
	 }
	 
	 
	 // Send new start condition and initiate read
	 returnval = i2cTransmitByte( (address7 | 0x01), I2C_SEND_START, I2C_NO_STOP );
	 
	 if( returnval != I2C_SUCCESS )
	 {
		  return i2cFailed(returnval);
	 }
	 
	 // Start read
	 for( i = 0; i < (bytesToRead - 1); i++ )
	 {
		  rxBuffer[i] = i2cReadByte( I2C_ACK_ENABLE, I2C_NO_STOP, &returnval );
		  if( returnval != I2C_SUCCESS )
		  {
				return i2cFailed(returnval);
		  }
	 }
	 
	 // Read the last byte
	 rxBuffer[bytesToRead - 1] = i2cReadByte( I2C_ACK_DISABLE, I2C_SEND_STOP, &returnval );
	 
	 if( returnval != I2C_SUCCESS )
	 {
		  return i2cFailed(returnval);
	 }
	 
	 return I2C_SUCCESS;
}
